/*:
 * @target MZ
 * @plugindesc [HS] Save/Load/キャンセル後に共通EVを自動実行。Spine再構築とADV節頭リスタートを決定論で行うフック集
 * @author HS
 *
 * @param AfterSaveCommonEvent
 * @text セーブ後 共通イベントID
 * @type common_event
 * @default 0
 * @desc セーブ成功→マップへ戻った時に実行（例：SpineRefresh）。
 *
 * @param AfterLoadCommonEvent
 * @text ロード後 共通イベントID
 * @type common_event
 * @default 0
 * @desc ロード成功→マップ起動時に実行（ADVでない場合に使用）。
 *
 * @param AfterCancelCommonEvent
 * @text キャンセル後 共通イベントID
 * @type common_event
 * @default 0
 * @desc Save/Load画面をキャンセルで閉じてマップに戻った時に実行（例：SpineRefresh）。
 *
 * @param AlwaysOnMapStart
 * @text マップ開始時に常に実行
 * @type boolean @on 実行する @off 実行しない
 * @default false
 *
 * @param MapStartCommonEvent
 * @text MapStart 共通イベントID
 * @type common_event
 * @default 0
 * @desc AlwaysOnMapStart=true のときに毎回実行（例：SpineRefresh）。
 *
 * @param AdvModeSwitch
 * @text ADVモード判定スイッチ
 * @type switch
 * @default 0
 * @desc ONの状態でロードされたら、ADV側のリスタートを優先実行。
 *
 * @param AdvRestartCommonEvent
 * @text ADV再スタート 共通イベントID
 * @type common_event
 * @default 0
 * @desc ADVモード時のロードで実行（節頭ジャンプ等）。未設定なら下記の変数方式を試みる。
 *
 * @param AdvEventIdVariable
 * @text ADV共通EVID 変数
 * @type variable
 * @default 0
 * @desc ここに「現在のADV共通イベントID」を入れておけば、ロード時にそのIDをreserveして冒頭から再開。
 *
 * @help
 * ■ 目的
 *  - Save成功/Load成功/キャンセル（戻る）で、指定の共通EVを自動投入
 *  - Map開始時に任意の共通EVを自動投入（任意）
 *  - ADVモードでのロード時は、ADV節頭リスタートを最優先
 *
 * ■ 推奨運用
 *  - 共通EV側で「Spine全消し→変数に従って必要分だけ生成→アニメ適用」を決定論で実施（隠す運用はNG）
 *  - ADV中は AdvModeSwitch=ON、かつ AdvRestartCommonEvent か AdvEventIdVariable を運用
 *  - 既存の“自動復帰/重複除去系プラグイン”はOFF
 */
(() => {
  const PN = "HS_SaveLoadHooks";
  const P = PluginManager.parameters(PN);

  const CE_SAVE   = Number(P.AfterSaveCommonEvent || 0);
  const CE_LOAD   = Number(P.AfterLoadCommonEvent || 0);
  const CE_CANCEL = Number(P.AfterCancelCommonEvent || 0);
  const ALWAYS    = String(P.AlwaysOnMapStart || "false") === "true";
  const CE_START  = Number(P.MapStartCommonEvent || 0);
  const SW_ADV    = Number(P.AdvModeSwitch || 0);
  const CE_ADV    = Number(P.AdvRestartCommonEvent || 0);
  const VAR_ADVID = Number(P.AdvEventIdVariable || 0);

  const sys = () => $gameSystem;
  const pushPending = (id) => {
    if (!id) return;
    sys()._hs_pendingCEs = sys()._hs_pendingCEs || [];
    sys()._hs_pendingCEs.push(id);
  };
  const shiftPending = () => (sys()._hs_pendingCEs && sys()._hs_pendingCEs.shift()) || 0;
  const hasPending = () => sys()._hs_pendingCEs && sys()._hs_pendingCEs.length > 0;

  // ---- Save成功：戻り後にCE_SAVE ----
  const _onSaveSuccess = Scene_Save.prototype.onSaveSuccess;
  Scene_Save.prototype.onSaveSuccess = function() {
    _onSaveSuccess.call(this);
    sys()._hs_fileResult = "save";
  };

  // ---- Load成功：Map起動時に ADV優先 or CE_LOAD を予約 ----
  const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
  Scene_Load.prototype.onLoadSuccess = function() {
    const advOn = SW_ADV > 0 ? $gameSwitches.value(SW_ADV) : false;
    if (advOn) {
      // ①固定のADVリスタートCEがあればそれを
      if (CE_ADV > 0) pushPending(CE_ADV);
      // ②無ければ変数で指示された共通EVIDを冒頭から
      else if (VAR_ADVID > 0) {
        const id = $gameVariables.value(VAR_ADVID) || 0;
        if (id > 0) pushPending(id);
      }
      // ③どちらも無ければ通常ロード後CE
      else if (CE_LOAD > 0) pushPending(CE_LOAD);
    } else {
      if (CE_LOAD > 0) pushPending(CE_LOAD);
    }
    sys()._hs_fileResult = "load";
    _onLoadSuccess.call(this);
  };

  // ---- Save/Loadキャンセル：戻り後にCE_CANCEL（未設定ならCE_SAVE→CE_STARTの順でフォールバック） ----
  const _file_onCancel = Scene_File.prototype.onCancel;
  Scene_File.prototype.onCancel = function() {
    // 直前にsave/load成功が確定していればキャンセル扱いは不要
    if (sys()._hs_fileResult !== "save" && sys()._hs_fileResult !== "load") {
      sys()._hs_fileResult = "cancel";
    }
    _file_onCancel.call(this);
  };

  // ---- Map.start：戻り種別を見てCEをキュー → 安全なフレームで順次投入 ----
  const _map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _map_start.call(this);

    if (ALWAYS && CE_START > 0) pushPending(CE_START);

    switch (sys()._hs_fileResult) {
      case "save":
        if (CE_SAVE > 0) pushPending(CE_SAVE);
        break;
      case "cancel":
        if (CE_CANCEL > 0) pushPending(CE_CANCEL);
        else if (CE_SAVE > 0) pushPending(CE_SAVE);
        else if (ALWAYS && CE_START > 0) pushPending(CE_START);
        break;
      // "load" は onLoadSuccess 側で既にpushPending済み
    }
    sys()._hs_fileResult = null;

    this._hs_cePumpTimer = 2; // 2f待ってから投入（演出レース対策）
  };

  const _map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
    _map_update.call(this);
    if (this._hs_cePumpTimer > 0) { this._hs_cePumpTimer--; return; }
    if (hasPending() && !$gameMap.isEventRunning() && !$gameMessage.isBusy()) {
      const nextId = shiftPending();
      if (nextId > 0) {
        // 連打貫通の芽を摘む（必要ならコメントアウト）
        Input.clear(); TouchInput.clear();
        $gameTemp.reserveCommonEvent(nextId);
      }
    }
  };
})();



